<?php

// This file is part of MyEducationPath module for Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * MyEducationPathmodule core interaction API
 *
 * @package    mod
 * @subpackage myeducationpath
 * @copyright  Roman Gelembjuk
 * @license    http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */

require_once($CFG->dirroot.'/course/lib.php');
require_once($CFG->libdir.'/gradelib.php');
require_once('lib/myeducationpath_wrapper.php');

define('_MYEDUCATIONPATH_ENDPOINT','myeducationpath.com');

/**
 * Add myedupath instance.
 *
 * @param stdClass $myedupath
 * @return int new myedupath instance id
 */
function myeducationpath_add_instance($myedupath) {
    global $DB,$CFG;

    $myedupath->timemodified = time();

    if ($returnid = $DB->insert_record('myeducationpath', $myedupath)) {
        $myedupath->id = $returnid;

        $event = new Object();
        $event->name        = $myedupath->name;
        $event->description = '';
        $event->courseid    = $myedupath->course;
        $event->groupid     = 0;
	$event->timestart   = 0;
        $event->userid      = 0;
        $event->eventtype   = 'course';
        $event->modulename  = 'myeducationpath';
        $event->instance    = $returnid;

	if(isset($CFG->branch) && $CFG->branch>='26'){

		 calendar_event::create($event);
	}else{
		 add_event($event);
	}
    }

    return $returnid;
}

/**
 * Update myedupath instance.
 *
 * @param stdClass $myedupath
 * @return bool true
 */
function myeducationpath_update_instance($myedupath) {
	global $DB,$CFG;

	$myedupath->timemodified = time();
	$myedupath->id = $myedupath->instance;

	if ($returnid = $DB->update_record('myeducationpath', $myedupath)) {
		$event = new Object();
		if ($event->id = $DB->get_field('event', 'id', array('modulename'=> 'myeducationpath', 'instance'=> $myedupath->id))) {
			$event->name        = $myedupath->name;
			if(isset($CFG->branch) && $CFG->branch>='26'){
				$calendarevent = calendar_event::load($event->id);
				$calendarevent->update($event);
			}else{
				update_event($event);
			}
		} else {
			$event = NULL;
			$event->name        = $myedupath->name;
			$event->description = '';
			$event->courseid    = $myedupath->course;
			$event->groupid     = 0;
			$event->userid      = 0;
			$event->timestart   = 0;
			$event->modulename  = 'myeducationpath';
			$event->instance    = $myedupath->id;

			if(isset($CFG->branch) && $CFG->branch>='26'){
				calendar_event::create($event);
			}else{
				add_event($event);
			}
		}
	} else {
		$DB->delete_records('event', array('modulename'=>'myeducationpath', 'instance'=> $myedupath->id));
	}

	return $returnid;
}

/**
 * Given an ID of an instance of this module,
 * this function will permanently delete the instance
 * and any data that depends on it.
 *
 * @param int $id
 * @return bool success
 */
function myeducationpath_delete_instance($id) {
    global $CFG, $DB, $USER;

    if (!$myedupath = $DB->get_record('myeducationpath', array('id'=> $id))) {
        return false;
    }

    // Prepare file record object
    if (!$cm = get_coursemodule_from_instance('myeducationpath', $id)) {
        return false;
    }

    $result = true;

    $DB->delete_records('myeducationpath_comments', array('activityid'=> $id));
    
    $DB->delete_records('myeducationpath_issues', array('certificateid'=> $id));
    
    if (!$DB->delete_records('myeducationpath', array('id'=> $id))) {
        $result = false;
    }

    return $result;
}

/**
 * @uses FEATURE_GROUPS
 * @uses FEATURE_GROUPINGS
 * @uses FEATURE_GROUPMEMBERSONLY
 * @uses FEATURE_MOD_INTRO
 * @uses FEATURE_COMPLETION_TRACKS_VIEWS
 * @uses FEATURE_GRADE_HAS_GRADE
 * @uses FEATURE_GRADE_OUTCOMES
 * @param string $feature FEATURE_xx constant for requested feature
 * @return mixed True if module supports feature, null if doesn't know
 */
function myeducationpath_supports($feature) {
    switch($feature) {
        //case FEATURE_GROUPS:                  return true;
        //case FEATURE_GROUPINGS:               return true;
        //case FEATURE_GROUPMEMBERSONLY:        return true;
        case FEATURE_MOD_INTRO:               return true;

        default: return null;
    }
}

function myeducationpath_get_context($id,&$error){
	global $DB,$CFG;
	if (! $cm = get_coursemodule_from_id('myeducationpath', $id)) {	
		$error='Course Module ID was incorrect';
		return array(null);
	}
	if (! $course = $DB->get_record('course', array('id'=> $cm->course))) {
		$error='course is misconfigured';
		return array(null);
	}
	if (! $mepinstance = $DB->get_record('myeducationpath', array('id'=> $cm->instance))) {
		$error='course module is incorrect';
		return array(null);
	}

	if(isset($CFG->branch) && $CFG->branch>='26'){
		$context = context_module::instance($cm->id);
	}else{
		$context = get_context_instance(CONTEXT_MODULE, $cm->id);
	}
	return array($cm,$course,$mepinstance,$context);
}
/**
 * Prints the header in view.  Used to help prevent FPDF header errors.
 * 
 * @param stdClass $course
 * @param stdClass $myedupath
 * @param stdClass $cm
 * @return null
 */
function myeducationpath_view_header($course, $mepinstance, $cm,$message='',$teacherview=false) {
	global $CFG, $DB, $PAGE, $OUTPUT;

	$PAGE->set_title(format_string($mepinstance->name));
	$PAGE->set_heading(format_string($course->fullname));
	echo $OUTPUT->header();

	if(isset($CFG->branch) && $CFG->branch>='26'){
		$context = context_module::instance($cm->id);
	}else{
		$context = get_context_instance(CONTEXT_MODULE, $cm->id);
	}

	if($message!=''){
		echo '<h4 style="color:#FF4D4D; margin:10px;">'.$message.'</h4>'."\n";
	}
	if (!empty($mepinstance->intro) && !$teacherview) {
		global $OUTPUT;

		echo $OUTPUT->box(format_text($mepinstance->intro), 'generalbox', 'intro');
	}
}

/**
 * Function to be run periodically according to the moodle cron
 * TODO:This needs to be done                        
 */
function myeducationpath_cron () {
    return true;
}

function myeducationpath_get_allcourse_with_activity(){
	global $DB,$CFG;
	$sql="SELECT distinct course,category1,category2,category3,tags,imageurl FROM {$CFG->prefix}myeducationpath";
	$courses = $DB->get_records_sql($sql);
	return $courses;
}

//
function myeducationpath_get_comments($course,$mepinstance){
	global $DB;
	$comments = $DB->get_records('myeducationpath_comments', array('activityid'=>$mepinstance->id));

	if(count($comments)==0) return array();
	$list=array();
	foreach($comments as $c){
		$user=$DB->get_record('user',array('id'=>$c->userid));
		$c->user=fullname($user);
		$list[]=$c;
	}
	return $list;
}
//========= CERTIFICATES MANAGE functions ======================================
//----- functions to reset certificates when a course was reset
function myeducationpath_reset_userdata($data) {
    global $CFG, $DB;

    $componentstr = get_string('modulenameplural', 'myeducationpath');
    $status = array();

    if (!empty($data->reset_certificate)) {
    	    //get all certificates for the course and delete them
    	    
    	    $instances = $DB->get_records('myeducationpath', array('course'=>$data->courseid));
    	    foreach($instances as $inst){
    	    	    $certificates = $DB->get_records('myeducationpath_issues', array('certificateid'=>$inst->id));
    	    	    foreach($certificates as $cert){
    	    	    	    myeducationpath_issue_delete($cert->id);
    	    	    }
    	    }
    	    $status[] = array('component'=>get_string('modulenameplural', 'myeducationpath'), 'item'=>get_string('certificateremoved', 'myeducationpath'), 'error'=>false);
    }
    return $status;
}

function myeducationpath_reset_course_form_definition(&$mform) {
    $mform->addElement('header', 'certificateheader', get_string('modulenameplural', 'myeducationpath'));
    $mform->addElement('advcheckbox', 'reset_certificate', get_string('deletissuedcertificates','myeducationpath'));
}

function myeducationpath_reset_course_form_defaults($course) {
    return array('reset_certificate'=>1);
}

/**
 * Returns information about received certificate.
 * Used for user activity reports.
 * 
 * @param stdClass $course
 * @param stdClass $user
 * @param stdClass $mod
 * @param stdClass $certificate
 * @return object|null
 */
function myeducationpath_user_outline($course, $user, $mod, $certificate) {
    global $DB;

    if ($issue = $DB->get_record('myeducationpath_issues', array('certificateid' => $certificate->id, 'userid' => $user->id))) {
        $result->info = get_string('issued', 'myeducationpath');
        $result->time = $issue->certdate;
    } else {
        $result->info = get_string('notissued', 'myeducationpath');
    }

    return $result;
}

/**
 * Returns information about received certificate.
 * Used for user activity reports.
 * 
 * @param stdClass $course
 * @param stdClass $user
 * @param stdClass $mod
 * @param stdClass $page
 * @return object|null
 */
function myeducationpath_user_complete($course, $user, $mod, $certificate) {
   global $DB;

   if ($issue = $DB->get_record('myeducationpath_issues', array('certificateid' => $certificate->id, 'userid' => $user->id))) {
        print_simple_box_start();
        echo get_string('issued', 'myeducationpath').": ";
        echo userdate($issue->certdate);
        echo '<br />';
        print_simple_box_end();
    } else {
        print_string('notissuedyet', 'myeducationpath');
    }
}

/**
 * Must return an array of user records (all data) who are participants
 * for a given instance of certificate.
 * 
 * @param int $certificateid
 * @return object list of participants
 */
function myeducationpath_get_participants($certificateid) {
    global $CFG, $DB;

    //Get students
    $participants = $DB->get_records_sql("SELECT DISTINCT u.id, u.id
                                          FROM {user} u,
                                               {myeducationpath_issues} a
                                          WHERE a.certificateid = '$certificateid' and
                                                u.id = a.userid");
    return $participants;
}


/**
 * Returns a list of issued certificates - sorted for report.
 * 
 * @param stdClass $certificate
 * @param string $sort the sort order
 * @param boolean $groupmode are we in group mode ?
 * @param stdClass $cm the course module
 */
function myeducationpath_get_issues($certificate, $sort="u.studentname ASC", $groupmode, $cm) {
	global $CFG, $DB;
	// get all users that can manage this certificate to exclude them from the report.
	if(isset($CFG->branch) && $CFG->branch>='26'){
		$context = context_module::instance($cm->id);
	}else{
		$context = get_context_instance(CONTEXT_MODULE, $cm->id);
	}
	$certmanagers = get_users_by_capability($context, 'mod/myeducationpath:manage', 'u.id');
	
	// get all the users that have certificates issued.
	$users = $DB->get_records_sql("SELECT u.*,u.picture, s.code, s.timecreated, s.certdate, s.studentname, s.reportgrade,s.cwlink,s.id as issueid
				   FROM {$CFG->prefix}myeducationpath_issues s,
					{$CFG->prefix}user u
				   WHERE s.certificateid = '$certificate'
				   AND s.userid = u.id
				   AND s.certdate > 0
				   GROUP BY u.id");
	// now exclude all the certmanagers.
	foreach ($users as $id => $user) {
	if (isset($certmanagers[$id])) { //exclude certmanagers.
	    unset($users[$id]);
	}
	}
	
	// if groupmembersonly used, remove users who are not in any group
	if (!empty($users) and !empty($CFG->enablegroupings) and $cm->groupmembersonly) {
	if ($groupingusers = groups_get_grouping_members($cm->groupingid, 'u.id', 'u.id')) {
	    $users = array_intersect($users, array_keys($groupingusers));
	}
	}
	
	if (!$groupmode) {
	return $users;
	} else {
	$currentgroup = groups_get_activity_group($cm);
	if ($currentgroup) {
	    $groupusers = groups_get_members($currentgroup, 'u.*');
	    if (empty($groupusers)) {
		return array();
	    }
	    foreach($groupusers as $id => $gpuser) {
		if (!isset($users[$id])) {
		    //remove this user as it isn't in the group!
		    unset($users[$id]);
		}
	    }
	}
	
	return $users;
	}
}

/**
 * Returns a list of all course students or all who don't have a certificate.
 * 
 * @param stdClass $certificate
 * @param stdClass $cm the course module
 */
function myeducationpath_get_course_students($certificate,$cm,$all=false,$groupmode=false){
	global $CFG, $DB;
	// get all users that can manage this certificate to exclude them from the report.

	if(isset($CFG->branch) && $CFG->branch>='26'){
		$context = context_module::instance($cm->id);
	}else{
		$context = get_context_instance(CONTEXT_MODULE, $cm->id);
	}
	$certmanagers = get_users_by_capability($context, 'mod/myeducationpath:manage', 'u.id');
	$managers=array_keys($certmanagers);

	$sql="SELECT u.* ".
		"FROM {$CFG->prefix}user u JOIN (".
			"SELECT DISTINCT eu1_u.id FROM {$CFG->prefix}user eu1_u JOIN {$CFG->prefix}user_enrolments eu1_ue ON eu1_ue.userid = eu1_u.id JOIN {$CFG->prefix}enrol eu1_e ".
			"ON (eu1_e.id = eu1_ue.enrolid AND eu1_e.courseid = $certificate->course) WHERE eu1_u.deleted = 0  ) e ".
			"ON e.id = u.id ";
	$wh=array();
	if(count($managers)>0)
		$wh[]="u.id NOT IN (".join(',',$managers).")";
	
	if(!$all){
		//get students with certificates
		$l=myeducationpath_get_issues($certificate->id, null, null, $cm);
		$issids=array_keys($l);
		if(count($issids)>0) $wh[]="u.id NOT IN (".join(',',$issids).")";
	}
	
	if(count($wh)>0) $sql.="WHERE ".join(' AND ',$wh); 
	
	$users = $DB->get_records_sql($sql);
	
	if (!empty($users) and !empty($CFG->enablegroupings) and $cm->groupmembersonly) {
		if ($groupingusers = groups_get_grouping_members($cm->groupingid, 'u.id', 'u.id')) {
			$users = array_intersect($users, array_keys($groupingusers));
		}
	}

	if (!$groupmode) {
		return $users;
	} else {
		$currentgroup = groups_get_activity_group($cm);
		if ($currentgroup) {
			$groupusers = groups_get_members($currentgroup, 'u.*');
			if (empty($groupusers)) {
				return array();
			}
			foreach($groupusers as $id => $gpuser) {
				if (!isset($users[$id])) {
					//remove this user as it isn't in the group!
					unset($users[$id]);
				}
			}
		}
		return $users;
	}
}

/**
 * Used for course participation report (in case certificate is added).
 * 
 * @return array
 */
function myeducationpath_get_view_actions() {
    return array('view','view all','view report');
}

/**
 * Used for course participation report (in case certificate is added).
 * 
 * @return array
 */
function myeducationpath_get_post_actions() {
    return array('received');
}

function myeducationpath_get_link_for_iframe($certrecord){
	if($certrecord->cwcertid>0 && $certrecord->cwuserid>0){
		$link=_MYEDUCATIONPATH_ENDPOINT.'index.php?page=public&id='.$certrecord->cwuserid.'&certificate='.$certrecord->cwcertid.'&tmpl=lempty&onlycertificate=y';
		return $link;
	}
	return '';
}

